package edu.montana.csci.csci468.parser;

import edu.montana.csci.csci468.parser.expressions.*;
import edu.montana.csci.csci468.parser.statements.*;
import edu.montana.csci.csci468.tokenizer.CatScriptTokenizer;
import edu.montana.csci.csci468.tokenizer.Token;
import edu.montana.csci.csci468.tokenizer.TokenList;
import edu.montana.csci.csci468.tokenizer.TokenType;

import static edu.montana.csci.csci468.tokenizer.TokenType.*;

import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;



public class CatScriptParser {

    SymbolTable symbolTable = new SymbolTable();

    private TokenList tokens;
    private FunctionDefinitionStatement currentFunctionDefinition;

    public CatScriptProgram parse(String source) {
        tokens = new CatScriptTokenizer(source).getTokens();

        // first parse an expression
        CatScriptProgram program = new CatScriptProgram();
        program.setStart(tokens.getCurrentToken());
        Expression expression = null;
        try {
            expression = parseExpression();
        } catch(RuntimeException re) {
            // ignore :)
        }
        if (expression == null || tokens.hasMoreTokens()) {
            tokens.reset();
            while (tokens.hasMoreTokens()) {
                program.addStatement(parseProgramStatement());
            }
        } else {
            program.setExpression(expression);
        }

        program.setEnd(tokens.getCurrentToken());
        return program;
    }

    public CatScriptProgram parseAsExpression(String source) {
        tokens = new CatScriptTokenizer(source).getTokens();
        CatScriptProgram program = new CatScriptProgram();
        program.setStart(tokens.getCurrentToken());
        Expression expression = parseExpression();
        program.setExpression(expression);
        program.setEnd(tokens.getCurrentToken());
        return program;
    }

    //============================================================
    //  Statements
    //============================================================

    private Statement parseProgramStatement() {
        Statement funcStatement = parseFunctionDefinition();

        if(funcStatement != null){
            return funcStatement;
        }
        return parseStatement();
    }



    private Statement parseStatement() {

        //Print statement
        Statement printStmt = parsePrintStatement();
        if (printStmt != null) {
            return printStmt;
        }

        //If statement
        Statement ifStmt = parseIfStatement();
        if (ifStmt != null) {
            return ifStmt;
        }

        //For statement
        Statement forStmt = parseForStatement();
        if (forStmt != null){
            return forStmt;
        }

        //Variable statement
        Statement varStmt = parseVariableStatement();
        if (varStmt != null){
            return  varStmt;
        }

        //parse assignment or function call
        Statement asgnOrFuncStmt = parseAssignmentOrFunctionCallStatement();
        if (asgnOrFuncStmt != null){
            return  asgnOrFuncStmt;
        }

        if(currentFunctionDefinition != null){
            Statement retStmt = parseReturnStatement();
            if(retStmt != null){
                return retStmt;
            }


        }

        return new SyntaxErrorStatement(tokens.consumeToken());
    }

    private Statement parseReturnStatement() {

        ReturnStatement returnStatement = new ReturnStatement();
        require(RETURN,returnStatement);
        returnStatement.setFunctionDefinition(currentFunctionDefinition);

        if(!tokens.match(RIGHT_BRACE)) {
            Expression optionalParse = parseExpression();

            if (optionalParse != null) {
                returnStatement.setExpression(optionalParse);
            }
        }

        return returnStatement;
    }

    private Statement parseAssignmentOrFunctionCallStatement() {
        if(tokens.match(IDENTIFIER)){
            Token identifier = tokens.consumeToken();

            //Parse assignment function
            if(tokens.matchAndConsume(EQUAL)){
                AssignmentStatement assignmentStatement = parseAssignmentStatement(identifier);
                return assignmentStatement;
            }

            //Function call
            else{
                FunctionCallExpression functionCallExpression = parseFunctionCallExpression(identifier);

                return new FunctionCallStatement(functionCallExpression);

            }


        }
        return null;
    }

    private AssignmentStatement parseAssignmentStatement(Token identifier) {
        AssignmentStatement assignmentStatement = new AssignmentStatement();
        assignmentStatement.setVariableName(identifier.getStringValue());
        assignmentStatement.setExpression(parseExpression());

        return assignmentStatement;

    }


    private Statement parseVariableStatement() {
        if(tokens.match(VAR)){

            VariableStatement variableStatement = new VariableStatement();
            variableStatement.setStart(tokens.consumeToken());
            String name = tokens.getCurrentToken().getStringValue();
            variableStatement.setVariableName(name);
            tokens.consumeToken();

            boolean listFlag = false;


            //Explicit type case
            if(tokens.matchAndConsume(COLON)){


                String stringValue = tokens.getCurrentToken().getStringValue();

                if(tokens.match(IDENTIFIER)){

                    //TODO find cattype with exisiting code


                    if (stringValue.equals("int")){
                        tokens.consumeToken();
                        variableStatement.setExplicitType(CatscriptType.INT);
                    }

                    else if(tokens.getCurrentToken().getStringValue().equals("object")){
                        tokens.consumeToken();
                        variableStatement.setExplicitType(CatscriptType.OBJECT);
                    }

                    else if(tokens.getCurrentToken().getStringValue().equals("string")){
                        tokens.consumeToken();
                        variableStatement.setExplicitType(CatscriptType.STRING);
                    }

                    else if(tokens.getCurrentToken().getStringValue().equals("bool")){
                        tokens.consumeToken();
                        variableStatement.setExplicitType(CatscriptType.BOOLEAN);
                    }

                    else if(tokens.getCurrentToken().getStringValue().equals("list")){
                        listFlag = true;
                        tokens.consumeToken();

                        require(LESS,variableStatement);

                        if (tokens.getCurrentToken().getStringValue().equals("int")){
                            tokens.consumeToken();
                            variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.INT));
                        }
                        else if(tokens.getCurrentToken().getStringValue().equals("object")){
                            tokens.consumeToken();
                            variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.OBJECT));
                        }
                        else if(tokens.getCurrentToken().getStringValue().equals("string")){
                            tokens.consumeToken();
                            variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.STRING));
                        }
                        else if(tokens.getCurrentToken().getStringValue().equals("bool")){
                            tokens.consumeToken();
                            variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.BOOLEAN));
                        }



                        require(GREATER,variableStatement);
                    }


                }
            }

            //Implicit
            else{


                tokens.matchAndConsume(EQUAL);

                //Case: matches an identifier
                if (tokens.match(IDENTIFIER)){
                    Expression expressionValue = parseExpression();
                    //Object value = symbolTable.getSymbolType(expressionValue.toString());


                    variableStatement.setExplicitType(CatscriptType.OBJECT);
                    variableStatement.setEnd(expressionValue.getEnd());
                    variableStatement.setExpression(expressionValue);

                    return variableStatement;

                }

                else{
                    Expression expressionValue = parseExpression();
                    CatscriptType implicitType = returnImplicitType(expressionValue);
                    variableStatement.setExplicitType(implicitType);
                    variableStatement.setEnd(expressionValue.getEnd());
                    variableStatement.setExpression(expressionValue);

                    return variableStatement;
                }



            }


            //Explicit
            if (tokens.matchAndConsume(EQUAL)){

                CatscriptType beforeAssignment = variableStatement.getExplicitType();

                Expression expressionValue = parseExpression();
                variableStatement.setEnd(expressionValue.getEnd());
                variableStatement.setExpression(expressionValue);

                if (beforeAssignment != expressionValue.getType() && beforeAssignment != CatscriptType.OBJECT && listFlag != true){
                    expressionValue.addError(ErrorType.INCOMPATIBLE_TYPES);
                }


                return variableStatement;
            }


            return null;



        }

        else{
            return null;
        }


    }



    //TODO: return to a type rather than explicit type
    private CatscriptType returnImplicitType(Expression expressionValue) {

        CatscriptType value = expressionValue.getType();

        if(value.toString().equals("int") ){
            return CatscriptType.INT;
        }

        else if(value.toString().equals("object") ){
            return CatscriptType.OBJECT;
        }

        else{
            return null;
        }

//            else if(tokens.getCurrentToken().getStringValue().equals("string")){
//                tokens.consumeToken();
//                variableStatement.setExplicitType(CatscriptType.STRING);
//            }
//
//            else if(tokens.getCurrentToken().getStringValue().equals("bool")){
//                tokens.consumeToken();
//                variableStatement.setExplicitType(CatscriptType.BOOLEAN);
//            }
//
//            else if(tokens.getCurrentToken().getStringValue().equals("list")){
//                tokens.consumeToken();
//
//                require(LESS,variableStatement);
//
//                if (tokens.getCurrentToken().getStringValue().equals("int")){
//                    tokens.consumeToken();
//                    variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.INT));
//                }
//                else if(tokens.getCurrentToken().getStringValue().equals("object")){
//                    tokens.consumeToken();
//                    variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.OBJECT));
//                }
//                else if(tokens.getCurrentToken().getStringValue().equals("string")){
//                    tokens.consumeToken();
//                    variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.STRING));
//                }
//                else if(tokens.getCurrentToken().getStringValue().equals("bool")){
//                    tokens.consumeToken();
//                    variableStatement.setExplicitType(CatscriptType.getListType(CatscriptType.BOOLEAN));
//                }




    }

    private Statement parseIfStatement() {
        if(tokens.match(IF)){
            Token ifStart = tokens.consumeToken();
            IfStatement ifStatement = new IfStatement();
            require(LEFT_PAREN,ifStatement);
            Expression testExpression = parseExpression();
            ifStatement.setExpression(testExpression);
            require(RIGHT_PAREN,ifStatement);

            require(LEFT_BRACE,ifStatement);

            List<Statement> stms = new ArrayList<>();
            while(tokens.hasMoreTokens() && !tokens.match(RIGHT_BRACE)){
                Statement stmt = parseStatement();
                stms.add(stmt);
            }
            require(RIGHT_BRACE, ifStatement);
            ifStatement.setTrueStatements(stms);


            //TODO Check right brace compatibility
            if(!tokens.match(EOF) && !tokens.match(RIGHT_BRACE)) {
                //Handling Else
                require(ELSE, ifStatement);

                Statement additionalIfStatement = parseIfStatement();

                if (additionalIfStatement == null) {

                    require(LEFT_BRACE, ifStatement);
                    List<Statement> elseStms = new ArrayList<>();
                    while (tokens.hasMoreTokens() && !tokens.match(RIGHT_BRACE)) {
                        Statement elseStmt = parseStatement();
                        elseStms.add(elseStmt);
                    }

                    require(RIGHT_BRACE, ifStatement);
                    ifStatement.setElseStatements(elseStms);

                    return ifStatement;
                }

                return null;
            }
            else{
                return ifStatement;
            }

        }
        else{
            return null;
        }

    }

    private Statement parsePrintStatement() {
        if (tokens.match(PRINT)) {

            PrintStatement printStatement = new PrintStatement();
            printStatement.setStart(tokens.consumeToken());

            require(LEFT_PAREN, printStatement);
            printStatement.setExpression(parseExpression());
            printStatement.setEnd(require(RIGHT_PAREN, printStatement));

            return printStatement;
        } else {
            return null;
        }
    }

    private Statement parseForStatement() {

        if (tokens.match(FOR)) {

            ForStatement forStatement = new ForStatement();
            forStatement.setStart(tokens.consumeToken());

            require(LEFT_PAREN, forStatement);



            forStatement.setVariableName(tokens.getCurrentToken().getStringValue());
            tokens.consumeToken();

            require(IN,forStatement);

            forStatement.setExpression(parseExpression());

            require(RIGHT_PAREN, forStatement);
            require(LEFT_BRACE, forStatement);

            //Parse statement

            List<Statement> stms = new ArrayList<>();
            while(tokens.hasMoreTokens() && !tokens.match(RIGHT_BRACE)){
                Statement stmt = parseStatement();
                stms.add(stmt);
            }
            forStatement.setEnd(require(RIGHT_BRACE, forStatement));
            forStatement.setBody(stms);

            return forStatement;
        } else {
            return null;
        }
    }


    private Statement parseFunctionDefinition() {
        if(tokens.match(FUNCTION)){
            Token name = tokens.consumeToken();
            FunctionDefinitionStatement funcDef = new FunctionDefinitionStatement();
            Token funcName = require(IDENTIFIER,funcDef);

            currentFunctionDefinition = funcDef;
            require(LEFT_PAREN,funcDef);


            //TODO Remove probably
            List<Parameter> paramList = new ArrayList<>();

            //Param List
            while (!tokens.match(RIGHT_PAREN)){



                //Comma Case
                if(tokens.matchAndConsume(COMMA)){
                    continue;
                }

                //Match Identifier
                else if(tokens.match(IDENTIFIER)){

                    //Set name
                    String paramName = tokens.getCurrentToken().getStringValue();
                    tokens.consumeToken();

                    if(tokens.matchAndConsume(COLON)){

                        funcDef.addParameter(paramName, praseTypeLiteral() );
                    }
                    else{


                        TypeLiteral typeLiteral = new TypeLiteral();
                        typeLiteral.setType(CatscriptType.OBJECT);

                        funcDef.addParameter(paramName, typeLiteral);
                    }

                }
            }

            //Consume Right Paren
            tokens.consumeToken();

            //Expression case
            if (tokens.matchAndConsume(COLON)){

                funcDef.setType(praseTypeLiteral());

            }


            //Function Body
            List<Statement> statementsList = new ArrayList<>();
            require(LEFT_BRACE,funcDef);
            while (!tokens.match(RIGHT_BRACE)){
                Statement stmt = parseStatement();
                statementsList.add(stmt);
            }

            //Consume Right Brace
            tokens.consumeToken();

            funcDef.setBody(statementsList);

            funcDef.setName(funcName.getStringValue());

            if (funcDef.getType() == null){
                funcDef.setType(null);
            }



            currentFunctionDefinition = null;
            symbolTable.registerSymbol(funcDef.getName(),funcDef.getType());
            return funcDef;


        }

        return null;
    }

    private TypeLiteral praseTypeLiteral() {
        TypeLiteral typeLiteralOut = new TypeLiteral();

        if (tokens.match(IDENTIFIER)) {

            if(tokens.getCurrentToken().getStringValue().equals("object")){
                tokens.consumeToken();
                typeLiteralOut.setType(CatscriptType.OBJECT);
                return typeLiteralOut;
            }

            else if(tokens.getCurrentToken().getStringValue().equals("int")){
                tokens.consumeToken();
                typeLiteralOut.setType(CatscriptType.INT);
                return typeLiteralOut;
            }

            else if(tokens.getCurrentToken().getStringValue().equals("bool")){
                tokens.consumeToken();
                typeLiteralOut.setType(CatscriptType.BOOLEAN);
                return typeLiteralOut;
            }

            else if(tokens.getCurrentToken().getStringValue().equals("string")){
                tokens.consumeToken();
                typeLiteralOut.setType(CatscriptType.STRING);
                return typeLiteralOut;
            }

            //TODO may have to redo
            else if(tokens.getCurrentToken().getStringValue().equals("list")){
                tokens.consumeToken();

                if (tokens.matchAndConsume(LESS)){

                    if (tokens.getCurrentToken().getStringValue().equals("int")){
                        tokens.consumeToken();
                        typeLiteralOut.setType(CatscriptType.OBJECT);
                        tokens.matchAndConsume(GREATER);
                        return typeLiteralOut;
                    }
                    else if(tokens.getCurrentToken().getStringValue().equals("object")){
                        tokens.consumeToken();
                        typeLiteralOut.setType(CatscriptType.OBJECT);
                        tokens.matchAndConsume(GREATER);
                        return typeLiteralOut;
                    }
                    else if(tokens.getCurrentToken().getStringValue().equals("string")){
                        tokens.consumeToken();
                        typeLiteralOut.setType(CatscriptType.STRING);
                        tokens.matchAndConsume(GREATER);
                        return typeLiteralOut;
                    }
                    else if(tokens.getCurrentToken().getStringValue().equals("bool")){
                        tokens.consumeToken();
                        typeLiteralOut.setType(CatscriptType.BOOLEAN);
                        tokens.matchAndConsume(GREATER);
                        return typeLiteralOut;
                    }

                }

                typeLiteralOut.setType(CatscriptType.OBJECT);
                return typeLiteralOut;
            }

            return null;


        }
        return null;
    }


    //============================================================
    //  Expressions
    //============================================================

    private Expression parseExpression() {
        return parseEqualityExpression();
    }

    private Expression parseEqualityExpression() {
        Expression expression = parseComparisonExpression();
        while (tokens.match(BANG_EQUAL, EQUAL_EQUAL)) {
            Token operator = tokens.consumeToken();
            final Expression rightHandSide = parseComparisonExpression();
            EqualityExpression equalityExpression = new EqualityExpression(operator, expression, rightHandSide);
            equalityExpression.setStart(expression.getStart());
            equalityExpression.setEnd(rightHandSide.getEnd());
            expression = equalityExpression;
        }
        return expression;
    }

    private Expression parseComparisonExpression() {
        Expression expression = parseAdditiveExpression();
        while (tokens.match(GREATER, GREATER_EQUAL,LESS,LESS_EQUAL)) {
            Token operator = tokens.consumeToken();
            final Expression rightHandSide = parseAdditiveExpression();
            ComparisonExpression comparisonExpression = new ComparisonExpression(operator, expression, rightHandSide);
            comparisonExpression.setStart(expression.getStart());
            comparisonExpression.setEnd(rightHandSide.getEnd());
            expression = comparisonExpression;
        }
        return expression;
    }

    private Expression parseAdditiveExpression() {
        Expression expression = parseFactorExpression();
        while (tokens.match(PLUS, MINUS)) {
            Token operator = tokens.consumeToken();
            final Expression rightHandSide = parseFactorExpression();
            AdditiveExpression additiveExpression = new AdditiveExpression(operator, expression, rightHandSide);
            additiveExpression.setStart(expression.getStart());
            additiveExpression.setEnd(rightHandSide.getEnd());
            expression = additiveExpression;
        }
        return expression;
    }

    private Expression parseFactorExpression() {
        Expression expression = parseUnaryExpression();
        while (tokens.match(SLASH, STAR)) {
            Token operator = tokens.consumeToken();
            final Expression rightHandSide = parseUnaryExpression();
            FactorExpression factorExpression = new FactorExpression(operator, expression, rightHandSide);
            factorExpression.setStart(expression.getStart());
            factorExpression.setEnd(rightHandSide.getEnd());
            expression = factorExpression;
        }
        return expression;
    }

    private Expression parseUnaryExpression() {
        if (tokens.match(MINUS, NOT)) {
            Token token = tokens.consumeToken();
            Expression rhs = parseUnaryExpression();
            UnaryExpression unaryExpression = new UnaryExpression(token, rhs);
            unaryExpression.setStart(token);
            unaryExpression.setEnd(rhs.getEnd());
            return unaryExpression;
        } else {
            return parsePrimaryExpression();
        }
    }

    private Expression parsePrimaryExpression() {

        //Integer literal
        if (tokens.match(INTEGER)) {
            Token integerToken = tokens.consumeToken();
            IntegerLiteralExpression integerExpression = new IntegerLiteralExpression(integerToken.getStringValue());
            integerExpression.setToken(integerToken);
            return integerExpression;
        }

        //String literal
        else if (tokens.match(STRING)) {
            Token stringToken = tokens.consumeToken();
            StringLiteralExpression stringExpression = new StringLiteralExpression(stringToken.getStringValue());
            stringExpression.setToken(stringToken);
            return stringExpression;
        }

        //Bool expression
        else if (tokens.match(TRUE)){
            Token boolToken = tokens.consumeToken();
            BooleanLiteralExpression boolExpression = new BooleanLiteralExpression(true);
            boolExpression.setToken(boolToken);
            return boolExpression;
        }

        //Bool expression
        else if (tokens.match(FALSE)){
            Token boolToken = tokens.consumeToken();
            BooleanLiteralExpression boolExpression = new BooleanLiteralExpression(false);
            boolExpression.setToken(boolToken);
            return boolExpression;
        }

        //Null literal
        else if (tokens.match(NULL)){
            Token nullToken = tokens.consumeToken();
            NullLiteralExpression nullExpression = new NullLiteralExpression();
            nullExpression.setToken(nullToken);
            return nullExpression;
        }

        //Parse identifier or function call
        else if (tokens.match(IDENTIFIER)) {
            Token identifierToken = tokens.consumeToken();

            //Function call
            if (tokens.match(LEFT_PAREN)) {

                Expression functCallExpression = parseFunctionCallExpression(identifierToken);

                return functCallExpression;
            }

            //Parse identifier
            else {

                IdentifierExpression identifierExpression = new IdentifierExpression(identifierToken.getStringValue());
                identifierExpression.setToken(identifierToken);
                return identifierExpression;

            }
        }

        //List literal expression
        else if (tokens.match(LEFT_BRACKET)) {

            //Consume left bracket
            Token startListToken = tokens.consumeToken();
            List<Expression> localList = new ArrayList<>();

            //Iterate through elements of list
            while(!tokens.match(RIGHT_BRACKET)){

                //Check if we have unclosed list
                if(tokens.match(EOF)){

                    ListLiteralExpression listExpression = new ListLiteralExpression(localList);
                    listExpression.setStart(startListToken);
                    listExpression.addError(ErrorType.UNTERMINATED_LIST);
                    return listExpression;
                }

                //Check if we have a comma
                if(tokens.match(COMMA)){
                    tokens.consumeToken();
                }

                //Parse an element of the list
                else {
                    Expression primaryExpression = parsePrimaryExpression();
                    localList.add(primaryExpression);
                }
            }

            //Reached a right bracket, return expression
            Token listEndToken = tokens.consumeToken();
            ListLiteralExpression listExpression = new ListLiteralExpression(localList);
            listExpression.setStart(startListToken);
            listExpression.setEnd(listEndToken);
            return listExpression;
        }

        //Paren expressions
        else if (tokens.match(LEFT_PAREN)){
            tokens.consumeToken();
            Expression expression = parseExpression();

            //Consume right paren
            tokens.consumeToken();

            //Return our parenthesized expression
            ParenthesizedExpression parenthesizedExpression = new ParenthesizedExpression(expression);
            return parenthesizedExpression;
        }

        else {
            SyntaxErrorExpression syntaxErrorExpression = new SyntaxErrorExpression(tokens.consumeToken());
            return syntaxErrorExpression;
        }
    }



    private FunctionCallExpression parseFunctionCallExpression(Token identifier) {

        //Left Paren
        tokens.consumeToken();
        List<Expression> localList = new ArrayList<>();





        while(!tokens.match(RIGHT_PAREN)){

            //Check if we have unclosed arg list
            if(tokens.match(EOF)){

                FunctionCallExpression functionCallExpression = new FunctionCallExpression(identifier.getStringValue(), localList);
                functionCallExpression.setStart(identifier);
                functionCallExpression.addError(ErrorType.UNTERMINATED_ARG_LIST);

                return functionCallExpression;
            }

            //Check if we have a comma
            if(tokens.match(COMMA)){
                tokens.consumeToken();
            }

            //Parse an element of the arg list
            else {
                Expression primaryExpression = parseExpression();
                localList.add(primaryExpression);
            }
        }

        //Right Paren
        tokens.consumeToken();

        FunctionCallExpression functionCallExpression = new FunctionCallExpression(identifier.getStringValue(),localList);
        return functionCallExpression;
    }


    //============================================================
    //  Parse Helpers
    //============================================================
    private Token require(TokenType type, ParseElement elt) {
        return require(type, elt, ErrorType.UNEXPECTED_TOKEN);
    }

    private Token require(TokenType type, ParseElement elt, ErrorType msg) {
        if(tokens.match(type)){
            return tokens.consumeToken();
        } else {
            elt.addError(msg, tokens.getCurrentToken());
            return tokens.getCurrentToken();
        }
    }

}
