package edu.montana.csci.csci468.parser.expressions;

import edu.montana.csci.csci468.bytecode.ByteCodeGenerator;
import edu.montana.csci.csci468.eval.CatscriptRuntime;
import edu.montana.csci.csci468.parser.CatscriptType;
import edu.montana.csci.csci468.parser.SymbolTable;
import edu.montana.csci.csci468.tokenizer.Token;
import edu.montana.csci.csci468.tokenizer.TokenType;
import org.objectweb.asm.Label;
import org.objectweb.asm.Opcodes;

import java.util.Objects;

public class EqualityExpression extends Expression {

    private final Token operator;
    private final Expression leftHandSide;
    private final Expression rightHandSide;

    public EqualityExpression(Token operator, Expression leftHandSide, Expression rightHandSide) {
        this.leftHandSide = addChild(leftHandSide);
        this.rightHandSide = addChild(rightHandSide);
        this.operator = operator;
    }

    public Expression getLeftHandSide() {
        return leftHandSide;
    }

    public Expression getRightHandSide() {
        return rightHandSide;
    }

    @Override
    public String toString() {
        return super.toString() + "[" + operator.getStringValue() + "]";
    }

    public boolean isEqual() {
        return operator.getType().equals(TokenType.EQUAL_EQUAL);
    }

    @Override
    public void validate(SymbolTable symbolTable) {
        leftHandSide.validate(symbolTable);
        rightHandSide.validate(symbolTable);
    }

    @Override
    public CatscriptType getType() {
        return CatscriptType.BOOLEAN;
    }

    //==============================================================
    // Implementation
    //==============================================================

    @Override
    public Object evaluate(CatscriptRuntime runtime) {

        Object lhsValue = leftHandSide.evaluate(runtime);
        Object rhsValue = rightHandSide.evaluate(runtime);

        //Handle null case
        if (leftHandSide.evaluate(runtime) == null || rightHandSide.evaluate(runtime) == null ){
            if (isEqual()) {
                return lhsValue == rhsValue;
            }
            else{
                return !(lhsValue == rhsValue);
            }

        }

        if (isEqual()) {
            return lhsValue.equals(rhsValue);
        }
        else {
            return !lhsValue.equals(rhsValue);
        }



    }

    @Override
    public void transpile(StringBuilder javascript) {
        super.transpile(javascript);
    }

    @Override
    public void compile(ByteCodeGenerator code) {

        //compile both sides and box
        getLeftHandSide().compile(code);
        box(code,getLeftHandSide().getType());
        getRightHandSide().compile(code);
        box(code,getRightHandSide().getType());

        Label pushFalse = new Label();
        Label pushTrue = new Label();

        code.addMethodInstruction(Opcodes.INVOKESTATIC, ByteCodeGenerator.internalNameFor(Objects.class),
                "equals", "(Ljava/lang/Object;Ljava/lang/Object;)Z");

        if(isEqual()){
            code.addJumpInstruction(Opcodes.IFNE, pushTrue);
        }

        //Deal with bang equal
        else{
            code.addJumpInstruction(Opcodes.IFEQ, pushTrue);
        }

        code.pushConstantOntoStack(false);
        code.addJumpInstruction(Opcodes.GOTO, pushFalse);

        code.addLabel(pushTrue);
        code.pushConstantOntoStack(true);

        code.addLabel(pushFalse);
    }


}
