package edu.montana.csci.csci468.parser.expressions;

import edu.montana.csci.csci468.bytecode.ByteCodeGenerator;
import edu.montana.csci.csci468.eval.CatscriptRuntime;
import edu.montana.csci.csci468.parser.CatscriptType;
import edu.montana.csci.csci468.parser.ErrorType;
import edu.montana.csci.csci468.parser.ParseError;
import edu.montana.csci.csci468.parser.SymbolTable;
import edu.montana.csci.csci468.tokenizer.Token;
import edu.montana.csci.csci468.tokenizer.TokenType;
import org.objectweb.asm.Label;
import org.objectweb.asm.Opcodes;

public class UnaryExpression extends Expression {

    private final Token operator;
    private final Expression rightHandSide;

    public UnaryExpression(Token operator, Expression rightHandSide) {
        this.rightHandSide = addChild(rightHandSide);
        this.operator = operator;
    }

    public Expression getRightHandSide() {
        return rightHandSide;
    }

    public boolean isMinus() {
        return operator.getType().equals(TokenType.MINUS);
    }

    public boolean isNot() {
        return !isMinus();
    }

    @Override
    public String toString() {
        return super.toString() + "[" + operator.getStringValue() + "]";
    }

    @Override
    public void validate(SymbolTable symbolTable) {
        rightHandSide.validate(symbolTable);
        if (isNot() && !rightHandSide.getType().equals(CatscriptType.BOOLEAN)) {
            addError(ErrorType.INCOMPATIBLE_TYPES);
        } else if(isMinus() && !rightHandSide.getType().equals(CatscriptType.INT)) {
            addError(ErrorType.INCOMPATIBLE_TYPES);
        }
    }

    @Override
    public CatscriptType getType() {
        if (isMinus()) {
            return CatscriptType.INT;
        } else {
            return CatscriptType.BOOLEAN;
        }
    }

    //==============================================================
    // Implementation
    //==============================================================

    @Override
    public Object evaluate(CatscriptRuntime runtime) {
        Object rhsValue = getRightHandSide().evaluate(runtime);
        if (this.isMinus()) {
            return -1 * (Integer) rhsValue;
        } else {

            return !((Boolean) (rhsValue));



        }
    }

    @Override
    public void transpile(StringBuilder javascript) {
       

        super.transpile(javascript);
    }

    @Override
    public void compile(ByteCodeGenerator code) {

        getRightHandSide().compile(code);

        Label falseLabel = new Label();
        Label trueLabel = new Label();

        //Minus case
        if(this.isMinus()) {
            code.pushConstantOntoStack(-1);
            code.addInstruction(Opcodes.IMUL);
            code.addJumpInstruction(Opcodes.GOTO,falseLabel);
        }

        //Not case
        else if (this.isNot()){
            code.addJumpInstruction(Opcodes.IFNE, trueLabel);
        }

        code.pushConstantOntoStack(true);
        code.addJumpInstruction(Opcodes.GOTO, falseLabel);

        code.addLabel(trueLabel);
        code.pushConstantOntoStack(false);

        code.addLabel(falseLabel);
    }


}
