package edu.montana.csci.csci468.parser;

import edu.montana.csci.csci468.parser.expressions.*;
import edu.montana.csci.csci468.parser.statements.*;
import edu.montana.csci.csci468.tokenizer.CatScriptTokenizer;
import edu.montana.csci.csci468.tokenizer.Token;
import edu.montana.csci.csci468.tokenizer.TokenList;
import edu.montana.csci.csci468.tokenizer.TokenType;

import java.util.ArrayList;
import java.util.List;

import static edu.montana.csci.csci468.tokenizer.TokenType.*;

public class CatScriptParser {

    private TokenList tokens;
    private FunctionDefinitionStatement currentFunctionDefinition;

    public CatScriptProgram parse(String source) {
        tokens = new CatScriptTokenizer(source).getTokens();

        // first parse an expression
        CatScriptProgram program = new CatScriptProgram();
        program.setStart(tokens.getCurrentToken());
        Expression expression = null;
        try {
            expression = parseExpression();
        } catch(RuntimeException re) {
            // ignore :)
        }
        if (expression == null || tokens.hasMoreTokens()) {
            tokens.reset();
            while (tokens.hasMoreTokens()) {
                program.addStatement(parseProgramStatement());
            }
        } else {
            program.setExpression(expression);
        }

        program.setEnd(tokens.getCurrentToken());
        return program;
    }

    public CatScriptProgram parseAsExpression(String source) {
        tokens = new CatScriptTokenizer(source).getTokens();
        CatScriptProgram program = new CatScriptProgram();
        program.setStart(tokens.getCurrentToken());
        Expression expression = parseExpression();
        program.setExpression(expression);
        program.setEnd(tokens.getCurrentToken());
        return program;
    }

    //============================================================
    //  Statements
    //============================================================

    private Statement parseProgramStatement() {
        if (tokens.match(PRINT)) {
            return parsePrintStatement();
        } else if (tokens.match(IF)) {
            return parseIfStatement();
        } else if(tokens.match(FOR)) {
            return parseForStatement();
        } else if(tokens.match(VAR)) {
            return  parseVarStatement();
        } else if (tokens.match(FUNCTION)) {
            return parseFunctionStatement();
        } else if(tokens.match(RETURN)) {
            return parseReturnStatement();
        } else if (tokens.match(IDENTIFIER)) {
            return parseFunctionCallStatement();
        }

        return new SyntaxErrorStatement(tokens.consumeToken());
    }

    private Statement parsePrintStatement() {
            PrintStatement printStatement = new PrintStatement();
            printStatement.setStart(tokens.consumeToken());

            require(LEFT_PAREN, printStatement);
            printStatement.setExpression(parseExpression());
            printStatement.setEnd(require(RIGHT_PAREN, printStatement));

            return printStatement;
    }

    private Statement parseIfStatement() {
        IfStatement ifs = new IfStatement();
        ifs.setStart(tokens.consumeToken());

        require(LEFT_PAREN, ifs);
        ifs.setExpression(parseExpression());
        require(RIGHT_PAREN, ifs);
        require(LEFT_BRACE, ifs);

        ArrayList<Statement> t = new ArrayList<>();
        ArrayList<Statement> e = new ArrayList<>();

        t.add(parseProgramStatement());
        require(RIGHT_BRACE, ifs);

        if (tokens.match(ELSE)) {
            tokens.consumeToken();
            require(LEFT_BRACE, ifs);
            if (!tokens.match(EOF))
                e.add(parseProgramStatement());
            require(RIGHT_BRACE, ifs);
        }

        ifs.setTrueStatements(t);
        ifs.setElseStatements(e);
        return ifs;
    }

    private Statement parseForStatement() {
        ForStatement forStatement = new ForStatement();
        forStatement.setStart(tokens.consumeToken());

        require(LEFT_PAREN, forStatement);
        forStatement.setVariableName(tokens.consumeToken().getStringValue());
        require(IN, forStatement);
        forStatement.setExpression(parseExpression());
        require(RIGHT_PAREN, forStatement);
        require(LEFT_BRACE, forStatement);

        ArrayList<Statement> body = new ArrayList<>();

        body.add(parseProgramStatement());

        forStatement.setBody(body);
        forStatement.setEnd(require(RIGHT_BRACE, forStatement));

        return forStatement;
    }

    private Statement parseVarStatement() {
        VariableStatement variableStatement = new VariableStatement();

        variableStatement.setStart(tokens.consumeToken());

        variableStatement.setVariableName(tokens.consumeToken().getStringValue());

        if (tokens.match(EQUAL)) {
            tokens.consumeToken();

            if (tokens.match(INTEGER)) {
                variableStatement.setExplicitType(CatscriptType.INT);
            } else if (tokens.match(NULL)) {
                variableStatement.setExplicitType(CatscriptType.NULL);
                return variableStatement;
            }

        } else if (tokens.match(COLON)) {
            tokens.consumeToken();
            String tok = tokens.consumeToken().getStringValue();
            if ("int".equals(tok)) variableStatement.setExplicitType(CatscriptType.INT);
            else if ("bool".equals(tok)) variableStatement.setExplicitType(CatscriptType.BOOLEAN);
            else if ("string".equals(tok)) variableStatement.setExplicitType(CatscriptType.STRING);
            else if ("object".equals(tok)) variableStatement.setExplicitType(CatscriptType.OBJECT);
            else if ("list".equals(tok)) {
                require(LESS, variableStatement);
                String listTypeString = tokens.consumeToken().getStringValue();
                variableStatement.setExplicitType(CatscriptType.getListType(strToType(listTypeString)));
                require(GREATER, variableStatement);
            }
            tokens.consumeToken();
        }

        variableStatement.setExpression(parseExpression());

        return variableStatement;
    }

    private Statement parseFunctionStatement() {
        FunctionDefinitionStatement fds = new FunctionDefinitionStatement();
        fds.setStart(tokens.getCurrentToken());
        tokens.consumeToken();
        fds.setName(tokens.consumeToken().getStringValue());

        TypeLiteral tl = new TypeLiteral();
        tl.setType(CatscriptType.OBJECT);
        fds.setType(tl);

        require(LEFT_PAREN, fds);

        while (!tokens.match(RIGHT_PAREN)) {
            String name = tokens.consumeToken().getStringValue();
            TypeLiteral t = new TypeLiteral();

            if (tokens.match(COLON)) {
                tokens.consumeToken();


                t.setType(strToType(tokens.consumeToken().getStringValue()));
            } else {
                t.setType(CatscriptType.OBJECT);
            }


            fds.addParameter(name, t);

            if (tokens.match(COMMA)) {
                tokens.consumeToken();
            }
        }

        require(RIGHT_PAREN, fds);
        require(LEFT_BRACE, fds);
        List<Statement> statements = new ArrayList<>();

        currentFunctionDefinition = fds;

        while (!tokens.match(RIGHT_BRACE)) {
            statements.add(parseProgramStatement());
        }

        //CHECK PARSE RETURN STATEMENT


        fds.setBody(statements);
        fds.setEnd(tokens.getCurrentToken());
        tokens.consumeToken();
        fds.setType(null);
        return fds;
    }

    private Statement parseFunctionCallStatement() {
        Token id = tokens.consumeToken();
        if (tokens.match(EQUAL)) {
            //
        } else {
            FunctionCallExpression fce = parseFunctionCallExpression();
            return new FunctionCallStatement(fce);
        }
        return null;
    }

    private Statement parseReturnStatement() {
        ReturnStatement rs = new ReturnStatement();
        rs.setFunctionDefinition(currentFunctionDefinition);
        rs.setStart(tokens.consumeToken());
//        rs.setExpression(parseExpression());
        return rs;
    }

    //============================================================
    //  Expressions
    //============================================================

    private Expression parseExpression() {
        // = comparison_expression { ("!=" | "==") comparison_expression };
        Expression expression = parseComparisonExpression();
        while (tokens.match(BANG_EQUAL, EQUAL_EQUAL)) {
            Token operator = tokens.consumeToken();
            final Expression rhs = parseComparisonExpression();
            EqualityExpression equalityExpression = new EqualityExpression(operator, expression, rhs);
            equalityExpression.setStart(expression.getStart());
            equalityExpression.setEnd(rhs.getEnd());
            expression = equalityExpression;
        }
        return expression;
    }

    private Expression parseComparisonExpression() {
        Expression expression = parseAdditiveExpression();
        while (tokens.match(GREATER, GREATER_EQUAL, LESS, LESS_EQUAL)) {
            Token operator = tokens.consumeToken();
            final Expression rhs = parseAdditiveExpression();
            ComparisonExpression comparisonExpression = new ComparisonExpression(operator, expression, rhs);
            comparisonExpression.setStart(expression.getStart());
            comparisonExpression.setEnd(rhs.getEnd());
            expression = comparisonExpression;
        }
        return expression;

    }

    private Expression parseAdditiveExpression() {
        Expression expression = parseFactorExpression();
        while (tokens.match(PLUS, MINUS)) {
            Token operator = tokens.consumeToken();
            final Expression rightHandSide = parseFactorExpression();
            AdditiveExpression additiveExpression = new AdditiveExpression(operator, expression, rightHandSide);
            additiveExpression.setStart(expression.getStart());
            additiveExpression.setEnd(rightHandSide.getEnd());
            expression = additiveExpression;
        }
        return expression;
    }

    private Expression parseFactorExpression() {
        Expression expression = parseUnaryExpression();
        while (tokens.match(SLASH, STAR)) {
            Token operator = tokens.consumeToken();
            final Expression rightHandSide = parseUnaryExpression();
            FactorExpression factorExpression = new FactorExpression(operator, expression, rightHandSide);
            factorExpression.setStart(expression.getStart());
            factorExpression.setEnd(rightHandSide.getEnd());
            expression = factorExpression;
        }
        return expression;
    }

    private Expression parseUnaryExpression() {
        if (tokens.match(MINUS, NOT)) {
            Token token = tokens.consumeToken();
            Expression rhs = parseUnaryExpression();
            UnaryExpression unaryExpression = new UnaryExpression(token, rhs);
            unaryExpression.setStart(token);
            unaryExpression.setEnd(rhs.getEnd());
            return unaryExpression;
        } else {
            return parsePrimaryExpression();
        }
    }

    private Expression parsePrimaryExpression() {
        if (tokens.match(INTEGER)) {
            Token integerToken = tokens.consumeToken();
            IntegerLiteralExpression integerExpression = new IntegerLiteralExpression(integerToken.getStringValue());
            integerExpression.setToken(integerToken);
            return integerExpression;
        } else if (tokens.match(STRING)){
            Token stringToken = tokens.consumeToken();
            StringLiteralExpression stringExpression = new StringLiteralExpression(stringToken.getStringValue());
            stringExpression.setToken(stringToken);
            return stringExpression;
        }else if (tokens.match(TRUE, FALSE)){
            Token booleanToken = tokens.consumeToken();
            BooleanLiteralExpression booleanExpression = new BooleanLiteralExpression(Boolean.parseBoolean(booleanToken.getStringValue()));
            booleanExpression.setToken(booleanToken);
            return booleanExpression;
        }else if (tokens.match(NULL)) {
            Token nullToken = tokens.consumeToken();
            NullLiteralExpression nullExpression = new NullLiteralExpression();
            nullExpression.setToken(nullToken);
            return nullExpression;
        } else if(tokens.match(IDENTIFIER)) {
            Token idToken = tokens.consumeToken();
            IdentifierExpression idExpression = new IdentifierExpression(idToken.getStringValue());
            idExpression.setToken(idToken);
            return idExpression;
        } else if(tokens.match(LEFT_BRACKET)) {
            tokens.consumeToken();
            return parseListLiteralExpression();
        } else if(tokens.match(FUNCTION)) {
            return parseFunctionCallExpression();
        } else if(tokens.match(LEFT_PAREN)) {
            tokens.consumeToken();
            return parseParenthesizedExpression();
        } else {
            SyntaxErrorExpression syntaxErrorExpression = new SyntaxErrorExpression(tokens.consumeToken());
            return syntaxErrorExpression;
        }
    }

    private Expression parseListLiteralExpression() {
        ArrayList<Expression> l = new ArrayList();

        // if empty list return
        if (tokens.match(RIGHT_BRACKET)) {
            tokens.consumeToken();
            return new ListLiteralExpression(l);
        }

        //grab first expression and loop through list
        l.add(parseExpression());
        while (tokens.match(COMMA)) {
            tokens.consumeToken();
            l.add(parseExpression());
        }
        // create list
        ListLiteralExpression lle = new ListLiteralExpression(l);

        // check for end of list and add error if not there
        if (!tokens.match(RIGHT_BRACKET)) {
            lle.addError(ErrorType.UNTERMINATED_LIST);
        } else {
            tokens.consumeToken();
        }
        return lle;
    }

    private Expression parseParenthesizedExpression(){
        return new ParenthesizedExpression(parseExpression());
    }

    private FunctionCallExpression parseFunctionCallExpression() {
        Token t = tokens.consumeToken();
        List<Expression> args = null;
        do {
            args.add(parseExpression());
        } while (tokens.matchAndConsume(COMMA));
        FunctionCallExpression fc = new FunctionCallExpression(t.getStringValue(), args);
        return fc;
    }

    //============================================================
    //  Parse Helpers
    //============================================================
    private Token require(TokenType type, ParseElement elt) {
        return require(type, elt, ErrorType.UNEXPECTED_TOKEN);
    }

    private Token require(TokenType type, ParseElement elt, ErrorType msg) {
        if(tokens.match(type)){
            return tokens.consumeToken();
        } else {
            elt.addError(msg, tokens.getCurrentToken());
            return tokens.getCurrentToken();
        }
    }

    private static CatscriptType strToType(String s){
        switch (s){
            case "int":
                return CatscriptType.INT;
            case "bool" :
                return CatscriptType.BOOLEAN;
            case "string":
                return CatscriptType.STRING;
            case "object":
                return CatscriptType.OBJECT;
        }
        return CatscriptType.VOID;
    }

}
