package FSAComponents;

//import fsa.machine.*;

import java.awt.geom.Point2D;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.StringTokenizer;

import org.jdom.*;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;


public class FSAFile
{
    private Document _doc;
    private DocType _docType;
    private Element _alphabetElement = null;
    private Element _descriptionElement = null;
    private Element _finalStatesElement;
    private Element _initialStateElement;
    private Element _root;
    private Element _statesElement = null;
    private Element _transitionsElement = null;
    private HashMap _stateToElementMap;
    private HashMap _transitionToElementMap;
    private HashMap _statePropertyMap;
    private SAXBuilder _builder;
    private StateListener _stateListener;
    private StateMachine _machine;
    private String _systemId;
    private SymbolListener _alphabetListener;
    private SymbolSet _alphabet;
    private TransitionListener _transitionListener;
    private static final HashSet _empty = new HashSet();

    public FSAFile(StateMachine machine)
    {
	_machine = machine;
	_alphabet = _machine.getAlphabet();
	_stateToElementMap = new HashMap();
	_transitionToElementMap = new HashMap();
	_statePropertyMap = new HashMap();

	_builder = new SAXBuilder(true);
	_builder.setIgnoringElementContentWhitespace(true);
	//_builder.setErrorHandler(new FileErrorHandler());

	_systemId = getClass().getResource("/xml/fsa.dtd").toString();
	_systemId = _systemId.substring(0, _systemId.lastIndexOf('/') + 1);

	_stateListener = new FileStateListener();
	_transitionListener = new FileTransitionListener();
	_alphabetListener = new FileAlphabetListener();

	createDocument();
    }

    public void loadFile(InputStream inputStream)
	throws org.xml.sax.SAXException, java.io.IOException
    {
	HashMap stateIDMap;

	reset();
	parseFile(inputStream);

	_machine.removeStateListener(_stateListener);
	_machine.removeTransitionListener(_transitionListener);
	_alphabet.removeSymbolListener(_alphabetListener);

	loadAlphabet();
	stateIDMap = loadStates();
	loadInitialState(stateIDMap);
	loadFinalStates(stateIDMap);
	loadTransitions(stateIDMap);

	_machine.addStateListener(_stateListener);
	_machine.addTransitionListener(_transitionListener);
	_alphabet.addSymbolListener(_alphabetListener);
    }

    public void saveFile(OutputStream outputStream)
	throws java.io.IOException
    {
	XMLOutputter outputter;

	outputter = new XMLOutputter();
	outputter.setEncoding("UTF-8");
	outputter.setIndent(true);
	//outputter.setLineSeparator("\n");
	outputter.setNewlines(true);
	outputter.output(_doc, outputStream);
	
	//outputStream.close();
    }

    private void createDocument()
    {
	Iterator iterator;
	State state;
	StateEvent se;
	Transition transition;
	TransitionEvent te;
       
	_root = new Element("fsa");
	_docType = new DocType("fsa", "fsa.dtd");
	_doc = new Document(_root, _docType);

	_descriptionElement = new Element("description");
	_descriptionElement.setText(_machine.getDescription());
	_root.addContent(_descriptionElement);

	_alphabetElement = new Element("alphabet");
	_alphabetElement.setText(_alphabet.getSymbolString());
	_root.addContent(_alphabetElement);

	_statesElement = new Element("states");
	_root.addContent(_statesElement);

	iterator = _machine.getStates();
	
        while(iterator.hasNext())
	{
	    state = (State)iterator.next();
	    se = new StateEvent(this, StateEvent.ADDED, state);
	    _stateListener.stateAdded(se);
	}

	_initialStateElement = new Element("initial_state");
	_root.addContent(_initialStateElement);
	_stateListener.startStateChange(null);

	_finalStatesElement = new Element("final_states");
	_root.addContent(_finalStatesElement);
        _stateListener.finalStateChange(null);

	_transitionsElement = new Element("transitions");
	_root.addContent(_transitionsElement);

	iterator = _machine.getTransitions();
	
	while(iterator.hasNext())
	{
	    transition = (Transition)iterator.next();
	    te = new TransitionEvent(this, TransitionEvent.ADDED, transition);
	    _transitionListener.transitionAdded(te);
	}

	_machine.addStateListener(_stateListener);
	_machine.addTransitionListener(_transitionListener);
	_alphabet.addSymbolListener(_alphabetListener);
    }	

    private void reset()
    {
	_machine.reset();

	_machine.removeStateListener(_stateListener);
	_machine.removeTransitionListener(_transitionListener);
	_alphabet.removeSymbolListener(_alphabetListener);

	_stateToElementMap.clear();
	_transitionToElementMap.clear();

	_machine.addStateListener(_stateListener);
	_machine.addTransitionListener(_transitionListener);
	_alphabet.addSymbolListener(_alphabetListener);
    }

    private void parseFile(InputStream inputStream)
	throws org.xml.sax.SAXException, java.io.IOException
    {
	try
	{
	    _doc = _builder.build(inputStream, _systemId);
	    _root = _doc.getRootElement();
	}
	catch(JDOMException jde)
	{
	    jde.printStackTrace();
	    java.lang.System.exit(1);
	}
    }

    private void loadDescription()
    {
	_descriptionElement = _root.getChild("description");
	_machine.setDescription(_descriptionElement.getText());
    }

    private void loadAlphabet()
    {
	int i;
	String symbols;

	_alphabetElement = _root.getChild("alphabet");

	symbols = _alphabetElement.getTextTrim();

	for(i = 0; i < symbols.length(); i++)
	{
	    if(!_alphabet.contains(symbols.charAt(i)))
	    {
		try
		{
		    _alphabet.addSymbol(symbols.charAt(i));
		}
		catch(Exception e)
		{
		    e.printStackTrace();
		}
	    }
	}
    }

    private HashMap loadStates()
    {
	Attribute idAttribute = null;
	Element stateElement, propertyElement;
	Iterator states, properties;
	State state;
	String propName;
	String propVal;
	HashMap stateMap;

	stateMap = new HashMap();
	
	_statesElement = _root.getChild("states");

	states = _statesElement.getChildren("state").iterator();
	
	while(states.hasNext())
	{
	    stateElement = (Element)states.next();

	    
	    idAttribute = stateElement.getAttribute("id");

	    state = new State();
	    stateMap.put(idAttribute.getValue(), state);

	    _machine.addState(state);

	    properties = stateElement.getChildren("property").iterator();

	    while(properties.hasNext())
	    {
		propertyElement = (Element)properties.next();
		mapStateProperty(state, propertyElement);

		propName = propertyElement.getAttributeValue("name");
		propVal = propertyElement.getText();

		state.setProperty(propName, propVal);
	    }

	    idAttribute.setValue("s" + state.getKey());

	    mapStateElement(state, stateElement);
	}

	return stateMap;
    }

    private void loadInitialState(HashMap stateMap)
    {
	Attribute attribute;
	State initialState;

	_initialStateElement = _root.getChild("initial_state");
	attribute = _initialStateElement.getAttribute("stateid");
	initialState = (State)stateMap.get(attribute.getValue());

	if(initialState != null)
	{
	    initialState.setStartState(true);
	}

	_stateListener.startStateChange(null);
    }

    private void loadFinalStates(HashMap stateMap)
    {
	Attribute attribute;
	State finalState;
	StringTokenizer strtok;
	String stateID;

	_finalStatesElement = _root.getChild("final_states");
	attribute = _finalStatesElement.getAttribute("stateids");

	strtok = new StringTokenizer(attribute.getValue(), " ");

	while(strtok.hasMoreTokens())
	{
	    stateID = strtok.nextToken();
	    finalState = (State)stateMap.get(stateID);
	    finalState.setFinalState(true);
	}

	_stateListener.finalStateChange(null);
    }

    private void loadTransitions(HashMap stateMap)
    {
	Attribute stateAttribute = null, idAttribute = null,
	          symbolAttribute = null, nextAttribute = null;
	Element transitionElement, propertyElement;
	Iterator transitions, properties;
	boolean boolValue;
	State startState, endState;
	Transition transition;
	String id;
	SymbolSet transitionSymbols;
	String symbol;

	_transitionsElement = _root.getChild("transitions");
	
	transitions = _transitionsElement.getChildren("transition").iterator();

	while(transitions.hasNext())
	{
	    transitionElement = (Element)transitions.next();
	    
	    idAttribute = transitionElement.getAttribute("id");
	    stateAttribute = transitionElement.getAttribute("state");
	    symbolAttribute = transitionElement.getAttribute("symbol");
	    nextAttribute = transitionElement.getAttribute("next_state");
	    
	    startState = (State)stateMap.get(stateAttribute.getValue());
	    endState = (State)stateMap.get(nextAttribute.getValue());

	    transition = _machine.getTransition(startState, endState);

	    if(transition == null)
	    {
		transition = new Transition(startState, endState);
		_machine.addTransition(transition);
	    }

	    symbol = symbolAttribute.getValue();

	    idAttribute.setValue("e" + transition.getKey() + symbol.charAt(0));
	    stateAttribute.setValue("s" + startState.getKey());
	    nextAttribute.setValue("s" + endState.getKey());

	    transitionSymbols = transition.getSymbolSet();

	    try
	    {
		transitionSymbols.addSymbol(symbol.charAt(0));
	    }
	    catch(Exception e)
	    {
		e.printStackTrace();
	    }


	    mapTransitionElement(transition, transitionElement);
	}
    }

    private void mapStateElement(State state, Element element)
    {
	_stateToElementMap.put(state, element);
    }

    private void unmapStateElement(State state)
    {
	_stateToElementMap.remove(state);
    }

    private Element getStateElement(State state)
    {
	return (Element)_stateToElementMap.get(state);
    }

    private void mapTransitionElement(Transition transition, Element element)
    {
	HashMap elements;
	Attribute symbolAttribute;
	String symbol;

	symbolAttribute = element.getAttribute("symbol");
	symbol = symbolAttribute.getValue();

	elements = (HashMap)_transitionToElementMap.get(transition);

	if(elements == null)
	{
	    elements = new HashMap();
	    _transitionToElementMap.put(transition, elements);
	}

	elements.put(symbol, element);
    }

    private void unmapTransitionElement(Transition transition, String symbol)
    {
	HashMap elements;

	elements = (HashMap)_transitionToElementMap.get(transition);
	elements.remove(symbol);
    }

    private void unmapTransitionElements(Transition transition)
    {
	_transitionToElementMap.remove(transition);
    }

    private Element getTransitionElement(Transition transition, String symbol)
    {
	HashMap elements;

	elements = (HashMap)_transitionToElementMap.get(transition);

	if(elements == null)
	{
	    return null;
	}

	return (Element)elements.get(symbol);
    }

    private Iterator getTransitionElements(Transition transition)
    {
	HashMap elements;

	elements = (HashMap)_transitionToElementMap.get(transition);

	if(elements == null)
	{
	    return _empty.iterator();
	}

	return elements.values().iterator();
    }

    private Iterator getTransitionMapSymbols(Transition transition)
    {
	HashMap elements;

	elements = (HashMap)_transitionToElementMap.get(transition);

	if(elements == null)
	{
	    return _empty.iterator();
	}

	return elements.keySet().iterator();
    }

    private void mapStateProperty(State state, Element element)
    {
	HashMap elements;
	Attribute nameAttribute;
	String propertyName;

	nameAttribute = element.getAttribute("name");
	propertyName = nameAttribute.getValue();

	elements = (HashMap)_statePropertyMap.get(state);
	
	if(elements == null)
	{
	    elements = new HashMap();
	    _statePropertyMap.put(state, elements);
	}

	elements.put(propertyName, element);
    }

    private Element getPropertyElement(State state, String propertyName)
    {
	HashMap elements;
	Element element = null;

	elements = (HashMap)_statePropertyMap.get(state);

	if(elements != null)
	{
	    element = (Element)elements.get(propertyName);
	}

	return element;
    }

    private void unmapStateProperties(State state)
    {
	_statePropertyMap.remove(state);
    }

	
    public StateMachine getStateMachine()
    {
	return _machine;
    }

    private class FileStateListener
	implements StateListener
    {
	public void stateAdded(StateEvent se)
	{
	    Element stateElement;
	    Element propertyElement;
	    Iterator propertyNames;
	    String propertyName;
	    State state;
	
	    state = se.getState();

	    stateElement = new Element("state");
	    stateElement.setAttribute("id", "s" + state.getKey());

	    propertyNames = state.getPropertyNames();

	    while(propertyNames.hasNext())
	    {
		propertyName = (String)propertyNames.next();

		propertyElement = new Element("property");
		propertyElement.setAttribute("name", propertyName);
		propertyElement.setText(state.getProperty(propertyName));

		stateElement.addContent(propertyElement);
		mapStateProperty(state, propertyElement);
	    }


	    _statesElement.addContent(stateElement);
	    mapStateElement(state, stateElement);
	}

	public void stateRemoved(StateEvent se)
	{
	    Element element;
	    State state;

	    state = se.getState();

	    element = getStateElement(state);
	    _statesElement.removeContent(element);

	    unmapStateElement(state);
	    unmapStateProperties(state);
	}

	public void startStateChange(StateEvent se)
	{
	    Element element;
	    State state;

	    state = _machine.getStartState();

	    if(state == null)
	    {
		_initialStateElement.setAttribute("stateid", "");
	    }
	    else
	    {
		_initialStateElement.setAttribute("stateid", "s" 
						  + state.getKey());
	    }
	}

	public void finalStateChange(StateEvent se)
	{
	    Element element;
	    Iterator finalStates;
	    State finalState;
	    StringBuffer stateList;

	    stateList = new StringBuffer();
	    finalStates = _machine.getFinalStates();

	    while(finalStates.hasNext())
	    {
		finalState = (State)finalStates.next();
		stateList.append("s" + finalState.getKey());

		if(finalStates.hasNext())
		{
		    stateList.append(" ");
		}
	    }

	    _finalStatesElement.setAttribute("stateids", stateList.toString());
	}

	public void statePropertyChanged(StatePropertyEvent se)
	{
	    Element propertyElement;
	    Element stateElement;
	    State state;
	    String name;


	    state = se.getState();
	    name = se.getPropertyName();
	
	    stateElement = getStateElement(state);
	    propertyElement = getPropertyElement(state, name);

	    if(propertyElement == null)
	    {
		propertyElement = new Element("property");
		stateElement.addContent(propertyElement);
		propertyElement.setAttribute("name", name);
		mapStateProperty(state, propertyElement);
	    }

	    propertyElement.setText(state.getProperty(name));
	}
    }

    private class FileTransitionListener
	implements TransitionListener
    {
	public void transitionAdded(TransitionEvent te)
	{
	    Character symbol;
	    Element element;
	    Iterator symbols;
	    String startKey;
	    String endKey;
	    Transition transition;

	    transition = te.getTransition();

	    startKey = transition.getSource().getKey();
	    endKey = transition.getDestination().getKey();

	    symbols = transition.getSymbolSet().getSymbols();

	    while(symbols.hasNext())
	    {
		symbol = (Character)symbols.next();

		element = new Element("transition");

		element.setAttribute("id", "e" + transition.getKey() + symbol.toString());
		element.setAttribute("state", "s" + startKey);
		element.setAttribute("symbol", symbol.toString());
		element.setAttribute("next_state", "s" + endKey);

		_transitionsElement.addContent(element);
		mapTransitionElement(transition, element);
	    }
	}

	public void transitionRemoved(TransitionEvent te)
	{
	    Character symbol;
	    Element element;
	    Iterator symbols;
	    Transition transition;

	    transition = te.getTransition();

	    symbols = transition.getSymbolSet().getSymbols();

	    while(symbols.hasNext())
	    {
		symbol = (Character)symbols.next();

		element = getTransitionElement(transition, symbol.toString());
		_transitionsElement.removeContent(element);
	    }

	    unmapTransitionElements(transition);
	}

	public void transitionSymbolsChanged(TransitionEvent te)
	{
	    Character symbol;
	    Element element;
	    Iterator symbols;
	    String startKey;
	    String endKey;
	    String mappedSymbol;
	    SymbolSet tranSymbols;
	    Transition transition;

	    transition = te.getTransition();
	    tranSymbols = transition.getSymbolSet();
	    
	    startKey = transition.getSource().getKey();
	    endKey = transition.getDestination().getKey();

	    symbols = getTransitionMapSymbols(transition);

	    while(symbols.hasNext())
	    {
		mappedSymbol = (String)symbols.next();

		if(!tranSymbols.contains(mappedSymbol.charAt(0)))
		{
		    element = getTransitionElement(transition, mappedSymbol);

		    _transitionsElement.removeContent(element);
		    unmapTransitionElement(transition, mappedSymbol);
		}
	    }
	
	    symbols = tranSymbols.getSymbols();

	    while(symbols.hasNext())
	    {
		symbol = (Character)symbols.next();

		element = getTransitionElement(transition, symbol.toString());

		if(element == null)
		{
		    element = new Element("transition");

		    element.setAttribute("id", "e" + transition.getKey() + symbol.toString());
		    element.setAttribute("state", "s" + startKey);
		    element.setAttribute("symbol", symbol.toString());
		    element.setAttribute("next_state", "s" + endKey);

		    _transitionsElement.addContent(element);
		    mapTransitionElement(transition, element);
		}
	    }
	}

	public void transitionMoved(TransitionMovedEvent tme)
	{
	}
    }


    private class FileAlphabetListener
	implements SymbolListener
    {
	public void symbolsAdded(SymbolEvent e)
	{
	    SymbolSet set = (SymbolSet)e.getSource();
	    _alphabetElement.setText(set.getSymbolString());
	}

	public void symbolsRemoved(SymbolEvent e)
	{
	    SymbolSet set = (SymbolSet)e.getSource();
	    _alphabetElement.setText(set.getSymbolString());
	}

	public void aboutToAddSymbols(VetoableSymbolEvent e)
	{
	}

	public void aboutToRemoveSymbols(VetoableSymbolEvent e)
	{
	}
    }
}
